<?php
/**
 * Core: OxyProps custom Autoload
 *
 * Allows classes autoload with non PSR-4 class names to
 * accomodate WordPress coding standards
 *
 * @link        https://oxyprops.com
 *
 * @package     OxyProps
 * @subpackage  Core
 * @author      Cédric Bontems <cedric@thewebforge.dev>
 * @since       1.4.0 Creation.
 * @since       1.5.0 Add support for traits.
 * @copyright   Copyright (c) 2022, Cédric Bontems
 * @license     https://www.gnu.org/licenses/gpl-2.0.html  GPLv2 or later
 */

// If this file is called directly, abort.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

define( 'OXYPROPS_AUTOLOAD_VERSION', '1.1.0' );

/**
 * Converts class name to file name.
 *
 * Generates a name matching WordPress coding standards for the class file
 * from the class name.
 *
 * @see https://www.php.net/manual/fr/function.spl-autoload-register.php
 *
 * @since 1.4.0 Creation.
 * @author Cédric Bontems <cedric@thewebforge.dev>
 *
 * @param string $class_name The name of a class.
 * @return string The matching file name.
 */
function oxyprops_class_2_file( string $class_name ): string {
	return strtolower( str_replace( '_', '-', $class_name ) );
}

/**
 * OxyProps custom autoloader callback.
 *
 * Loads Classes and Traits files after generating their filenames from the
 * class name.
 *
 * @see https://github.com/php-fig/fig-standards/blob/master/accepted/PSR-4-autoloader-examples.md
 *
 * @since 1.4.0 Creation.
 * @since 1.5.0 Add trait support.
 * @author Cédric Bontems <cedric@thewebforge.dev>
 *
 * @param mixed $class The class to be loaded.
 * @return void
 */
function oxyprops_autoloader( $class ): void {

	// project-specific namespace prefix.
	$prefix = 'OxyProps\\Inc\\';

	// base directory for the namespace prefix.
	$base_dir = dirname( __DIR__ ) . '/includes/';

	// does the class use the namespace prefix?
	$len = strlen( $prefix );
	if ( strncmp( $prefix, $class, $len ) !== 0 ) {
		// no, move to the next registered autoloader.
		return;
	}

	// get the relative class name.
	$relative_class = 'class-' . oxyprops_class_2_file( substr( $class, $len ) );
	$class_file     = $base_dir . str_replace( '\\', '/', $relative_class ) . '.php';

	// get the relative trait name.
	$relative_trait = 'trait-' . oxyprops_class_2_file( substr( $class, $len ) );
	$trait_file     = $base_dir . str_replace( '\\', '/', $relative_trait ) . '.php';

	// if the file exists, require it.
	if ( file_exists( $class_file ) ) {
		require $class_file;
	}
	// if the file exists, require it.
	if ( file_exists( $trait_file ) ) {
		require $trait_file;
	}
}

spl_autoload_register( 'oxyprops_autoloader' );
